DROP PROCEDURE [hips].[PatientMasterMerge]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

-- =============================================
-- Author:        Simon Biber
-- Create date: 11 April 2013
-- Description:   Merges two records in the Patient Master Table
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterMerge] 
(
	@FromPatientMasterId INT,
	@ToPatientMasterId INT,
	@UserModified varchar(256)
)
AS
BEGIN
    SET NOCOUNT ON
    
    DECLARE @intErrorCode INT
		,@FromIhi VARCHAR(16)
		,@ToIhi VARCHAR(16)	
		,@ConflictStatus INT
		,@UnknownStatus INT
		,@Message VARCHAR(512)
		,@FromStatePatientId VARCHAR(64)
		,@ToStatePatientId VARCHAR(64)
		,@FromMrn VARCHAR(20)
		,@ToMrn VARCHAR(20)
		,@HospitalDesc VARCHAR(60)

    IF @ToPatientMasterId = @FromPatientMasterId
    BEGIN
		SET @intErrorCode = 50002
		RAISERROR @intErrorCode 'You cannot merge a patient with itself!'
		RETURN(@intErrorCode)
	END	
		
    SET @intErrorCode = 0
    SET @ConflictStatus = 102
    SET @UnknownStatus = -1

	-- Retrieve the IHI and SAUHI from both records
	SET @FromIhi = (SELECT [Ihi]
	FROM [hips].[PatientMasterIhi]
	WHERE [PatientMasterId]	= @FromPatientMasterId)
	
	SET @ToIhi = (SELECT [Ihi]
	FROM [hips].[PatientMasterIhi]
	WHERE [PatientMasterId]	= @ToPatientMasterId)
	
	SET @FromStatePatientId = (SELECT [StatePatientId]
	FROM [hips].[PatientMaster]
	WHERE [PatientMasterId] = @FromPatientMasterId)

	SET @ToStatePatientId = (SELECT [StatePatientId]
	FROM [hips].[PatientMaster]
	WHERE [PatientMasterId] = @ToPatientMasterId)

	-- Find out whether there are any MRNs from the same hospital on both patient masters.
	-- This should not happen, because the MRNs should have been merged in the PAS first.
	-- If there is a case of this occurring, grab the most recently updated MRN from each.
	-- This will be used to create a merge conflict alert if the IHI's are different.
	SELECT	 @FromMrn = hp1.Mrn
			,@ToMrn = hp2.Mrn
			,@HospitalDesc = h.[Description]
	FROM	hips.HospitalPatient hp1
	INNER JOIN hips.HospitalPatient hp2 ON hp1.HospitalId = hp2.HospitalId
	INNER JOIN hips.Hospital h on h.HospitalId = hp1.HospitalId
	WHERE	hp1.PatientMasterId = @FromPatientMasterId
	AND		hp2.PatientMasterId = @ToPatientMasterId
	ORDER BY hp1.DateModified DESC, hp2.DateModified DESC

	-- Move all clinical data (but not demographic data) from the non-surviving to the surviving patient master.
	UPDATE [hips].[HospitalPatient]
	SET [PatientMasterId] = @ToPatientMasterId
	WHERE [PatientMasterId] = @FromPatientMasterId	

	UPDATE [hips].[DownloadedDocument]
	SET [PatientMasterId] = @ToPatientMasterId
	WHERE [PatientMasterId] = @FromPatientMasterId	

	
	IF @FromIhi IS NOT NULL
	BEGIN
		IF @ToIhi IS NOT NULL
		BEGIN
			IF @FromIhi = @ToIhi
			BEGIN
				-- The IHIs match, this should only happen if one is marked as a duplicate.
				-- Remove the IHI from the non-surviving patient master
				DELETE [hips].[PatientMasterIhi]
				WHERE [PatientMasterId] = @FromPatientMasterId
				
				-- Remove the alert status, set it back to unknown. HIPS will revalidate and if OK set to Active.
				UPDATE [hips].[PatientMasterIhi]
				SET [IhiStatusId] = @UnknownStatus
				WHERE [PatientMasterId] = @ToPatientMasterId
			END
			ELSE IF (@FromMrn IS NOT NULL AND @ToMrn IS NOT NULL)
			BEGIN
				-- There are two different IHI's on the patient masters!
				-- Also, there is an MRN on each PM, from the same hospital!
				-- This means that a PAS merge should have been done before this enterprise merge.
				-- Create IHI Conflict alert against both patient masters.
				UPDATE [hips].[PatientMasterIhi]
				SET [IhiStatusId] = @ConflictStatus
					,[DateModified] = GETDATE()
					,[UserModified] = @UserModified
				WHERE [PatientMasterId]	= @FromPatientMasterId
				OR	[PatientMasterId] = @ToPatientMasterId
						
				SET @Message = 'IHI conflict merging SAUHI ' 
				    + @FromStatePatientId + ' having MRN '	+ @FromMrn + ' at ' + @HospitalDesc + ' and IHI ' + @FromIhi + ' into SAUHI '
					+ @ToStatePatientId + ' having MRN ' + @ToMrn + ' at ' + @HospitalDesc + ' and IHI ' + @ToIhi
				
				INSERT INTO [hips].IhiLookupAlert
						 ( PatientMasterId ,
						   Message ,
						   Request ,
						   Response ,
						   DateModified ,
						   UserModified ,
						   DateCreated ,
						   UserCreated
						 )
				 VALUES  ( @FromPatientMasterId , -- PatientMasterId - int
						   @Message, -- Message - varchar(512)
						   NULL, -- Request - xml
						   NULL, -- Response - xml
						   GETDATE() , -- DateModified - datetime
						   @UserModified , -- UserModified - varchar(256)
						   GETDATE() , -- DateCreated - datetime
						   @UserModified  -- UserCreated - varchar(256)
						 )
				INSERT INTO [hips].IhiLookupAlert
						 ( PatientMasterId ,
						   Message ,
						   Request ,
						   Response ,
						   DateModified ,
						   UserModified ,
						   DateCreated ,
						   UserCreated
						 )
				 VALUES  ( @ToPatientMasterId , -- PatientMasterId - int
						   @Message, -- Message - varchar(512)
						   NULL, -- Request - xml
						   NULL, -- Response - xml
						   GETDATE() , -- DateModified - datetime
						   @UserModified , -- UserModified - varchar(256)
						   GETDATE() , -- DateCreated - datetime
						   @UserModified  -- UserCreated - varchar(256)
						 )
			END
		END
		ELSE
		BEGIN
			-- The survivor's IHI field is blank.
			-- Transfer the IHI from the non-survivor to the survivor.

			-- 1. Remove the survivor's IHI information
			DELETE hips.PatientMasterIhi
			WHERE PatientMasterId = @ToPatientMasterId

			-- 2. Copy the IHI to the survivor
			INSERT hips.PatientMasterIhi
			SELECT @ToPatientMasterId
				,fromIhi.Ihi
				,fromIhi.IhiStatusId
				,fromIhi.IhiRecordStatusId
				,fromIhi.RegisteredGivenName
				,fromIhi.RegisteredFamilyName
				,fromIhi.RegisteredSexId
				,fromIhi.DateLastValidated
				,GETDATE()
				,@UserModified
				,GETDATE()
				,@UserModified
			FROM hips.PatientMasterIhi fromIhi
			WHERE fromIhi.PatientMasterId = @FromPatientMasterId
			
			-- 3. Remove the IHI from the non-survivor
			DELETE hips.PatientMasterIhi
			WHERE PatientMasterId = @FromPatientMasterId
		END
	END
	ELSE
	BEGIN
		-- The non-survivor's IHI field is blank.
				
		-- Remove the alert status from the survivor, set it back to unknown. 
		-- HIPS will revalidate and if OK set to Active.
		UPDATE [hips].[PatientMasterIhi]
		SET [IhiStatusId] = @UnknownStatus
		WHERE [PatientMasterId] = @ToPatientMasterId
	END

    SELECT @intErrorCode = @@ERROR 
    IF @intErrorCode = 0
    BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
		     , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]         
	     FROM	[hips].[PatientMaster] AS pm
         LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterId] = pm.[PatientMasterId]
         LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		 LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId
        WHERE	pm.[PatientMasterID] = @ToPatientMasterId
    END

    RETURN(@intErrorCode)
END
GO

